<?php
declare(strict_types=1);
mb_internal_encoding('UTF-8');
error_reporting(E_ALL);
ini_set('display_errors', '1');

date_default_timezone_set('America/Sao_Paulo');

/**
 * Consulta do resultado do Lote GNRE (robusta, com tentativas automáticas)
 * Uso: gnre_consulta_lote.php?recibo=25000225744095&ambiente=1
 *   ambiente: 1=PRODUÇÃO, 2=HOMOLOGAÇÃO
 */

/* ========= CERTIFICADO ========= */
$pfxPath     = __DIR__ . '/../photopaper.pfx';
$pfxPassword = 'photo008A';

/* ========= ENTRADA ========= */
$recibo   = isset($_GET['recibo']) ? preg_replace('/\D+/', '', $_GET['recibo']) : '';
$ambiente = isset($_GET['ambiente']) ? (int)$_GET['ambiente'] : 1; // 1 prod, 2 homolog

if ($recibo === '' || !in_array($ambiente, [1,2], true)) {
    http_response_code(400);
    echo "Parâmetros inválidos. Exemplo: ?recibo=25000225744095&ambiente=1";
    exit;
}

/* ========= ENDPOINT ========= */
$serviceNS = 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote';
$url = ($ambiente === 1)
    ? 'https://www.gnre.pe.gov.br/gnreWS/services/GnreResultadoLote'
    : 'https://www.testegnre.pe.gov.br:444/gnreWS/services/GnreResultadoLote';

/* ========= LOG ========= */
$baseDir   = dirname(__DIR__, 1);
$logDir    = $baseDir . '/logs';
if (!is_dir($logDir)) { @mkdir($logDir, 0775, true); }
$ts        = date('Ymd_His');
$logBase   = $logDir . "/consultaLote_{$recibo}_{$ts}";

/* ========= PAYLOADS DE DADOS ========= */
// v1 (pedido): TConsLote_GNRE (sem atributo versao)
$xml_v1 = <<<XML
<TConsLote_GNRE xmlns="http://www.gnre.pe.gov.br">
    <ambiente>{$ambiente}</ambiente>
    <numeroRecibo>{$recibo}</numeroRecibo>
</TConsLote_GNRE>
XML;

// v2 (pedido alternativo usado por algumas UF): TLote_ConsGNRE versao="2.00"
$xml_v2 = <<<XML
<TLote_ConsGNRE xmlns="http://www.gnre.pe.gov.br" versao="2.00">
    <ambiente>{$ambiente}</ambiente>
    <numeroRecibo>{$recibo}</numeroRecibo>
</TLote_ConsGNRE>
XML;

/* ========= CONSTRUTORES DE ENVELOPE ========= */
$buildSoap12Bare = function (string $serviceNS, string $xmlDados): string {
    return <<<SOAP
<?xml version="1.0" encoding="UTF-8"?>
<soap12:Envelope xmlns:soap12="http://www.w3.org/2003/05/soap-envelope"
                 xmlns:ws="{$serviceNS}">
  <soap12:Header>
    <ws:gnreCabecMsg>
      <ws:versaoDados>2.00</ws:versaoDados>
    </ws:gnreCabecMsg>
  </soap12:Header>
  <soap12:Body>
    <ws:gnreDadosMsg>
{$xmlDados}
    </ws:gnreDadosMsg>
  </soap12:Body>
</soap12:Envelope>
SOAP;
};

$buildSoap12Wrapped = function (string $serviceNS, string $xmlDados): string {
    return <<<SOAP
<?xml version="1.0" encoding="UTF-8"?>
<soap12:Envelope xmlns:soap12="http://www.w3.org/2003/05/soap-envelope"
                 xmlns:ws="{$serviceNS}">
  <soap12:Header>
    <ws:gnreCabecMsg>
      <ws:versaoDados>2.00</ws:versaoDados>
    </ws:gnreCabecMsg>
  </soap12:Header>
  <soap12:Body>
    <ws:consultar>
      <ws:gnreDadosMsg>
{$xmlDados}
      </ws:gnreDadosMsg>
    </ws:consultar>
  </soap12:Body>
</soap12:Envelope>
SOAP;
};

$buildSoap11Wrapped = function (string $serviceNS, string $xmlDados): string {
    return <<<SOAP
<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="{$serviceNS}">
  <soapenv:Header>
    <ws:gnreCabecMsg>
      <ws:versaoDados>2.00</ws:versaoDados>
    </ws:gnreCabecMsg>
  </soapenv:Header>
  <soapenv:Body>
    <ws:consultar>
      <ws:gnreDadosMsg>
{$xmlDados}
      </ws:gnreDadosMsg>
    </ws:consultar>
  </soapenv:Body>
</soapenv:Envelope>
SOAP;
};

/* ========= TENTATIVAS EM ORDEM ========= */
$attempts = [
    // [label, envelopeBuilder, xmlDados, contentType, soapAction]
    ['soap12_bare_v1',    $buildSoap12Bare,    $xml_v1, 'application/soap+xml; charset=utf-8; action="http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar"', 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar'],
    ['soap12_wrapped_v1', $buildSoap12Wrapped, $xml_v1, 'application/soap+xml; charset=utf-8; action="http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar"', 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar'],
    ['soap12_wrapped_v2', $buildSoap12Wrapped, $xml_v2, 'application/soap+xml; charset=utf-8; action="http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar"', 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar'],
    ['soap11_wrapped_v1', $buildSoap11Wrapped, $xml_v1, 'text/xml; charset=utf-8', 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar'],
];

$finalResponse = null;
$finalHttp     = 0;
$finalLabel    = '';
$finalReq      = '';

foreach ($attempts as [$label, $builder, $xmlDados, $contentType, $soapAction]) {

    $soapEnvelope = $builder($serviceNS, $xmlDados);

    // Log request
    file_put_contents("{$logBase}_{$label}_enviado.xml", $soapEnvelope);

    // cURL
    $ch = curl_init($url);
    $headers = [
        "Content-Type: {$contentType}",
        "SOAPAction: \"{$soapAction}\"",
    ];
    curl_setopt_array($ch, [
        CURLOPT_POST            => true,
        CURLOPT_POSTFIELDS      => $soapEnvelope,
        CURLOPT_HTTPHEADER      => $headers,
        CURLOPT_RETURNTRANSFER  => true,
        CURLOPT_HEADER          => false,
        CURLOPT_TIMEOUT         => 60,
        CURLOPT_SSL_VERIFYPEER  => true,
        CURLOPT_SSL_VERIFYHOST  => 2,
        CURLOPT_SSLCERT         => $pfxPath,
        CURLOPT_SSLCERTTYPE     => 'P12',
        CURLOPT_SSLCERTPASSWD   => $pfxPassword,
    ]);

    $response = curl_exec($ch);
    $errno    = curl_errno($ch);
    $error    = curl_error($ch);
    $httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Log response
    file_put_contents("{$logBase}_{$label}_recebido.xml", $response ?: $error);

    // Heurística de sucesso vs erro schema
    $isSchemaError = false;
    if ($response) {
        // Erros típicos
        if (strpos($response, '<ns1:codigo>199</ns1:codigo>') !== false ||
            strpos($response, '<ns1:codigo>501</ns1:codigo>') !== false ||
            stripos($response, 'Cannot find the declaration') !== false) {
            $isSchemaError = true;
        }
        // Sucesso mínimo: retorno TResultLote_GNRE com algo além do bloco de erro
        if (!$isSchemaError && (strpos($response, 'TResultLote_GNRE') !== false)) {
            $finalResponse = $response;
            $finalHttp     = $httpCode;
            $finalLabel    = $label;
            $finalReq      = $soapEnvelope;
            break;
        }
    }

    // Se 500/NoSOAPAction/NoService, segue tentando próximo
    if ($errno !== 0) {
        // continua loop
    }
}

// Saída na tela com diagnóstico
header('Content-Type: text/plain; charset=utf-8');

if ($finalResponse) {
    echo "✓ Variante que funcionou: {$finalLabel}\n";
    echo "URL: {$url}\nHTTP: {$finalHttp}\n\n";
    echo "XML Enviado ({$finalLabel}):\n{$finalReq}\n\n";
    echo "XML Recebido:\n{$finalResponse}\n";
} else {
    // Nenhuma variante passou: mostrar última resposta para depuração
    $last = end($attempts);
    $lastLabel = $last[0];
    $lastReq = @file_get_contents("{$logBase}_{$lastLabel}_enviado.xml");
    $lastResp = @file_get_contents("{$logBase}_{$lastLabel}_recebido.xml");

    echo "Nenhuma variante foi aceita pelo serviço.\n";
    echo "Endpoint: {$url}\n";
    echo "Última tentativa: {$lastLabel}\n\n";
    echo "XML Enviado (última):\n" . ($lastReq ?: '(sem)') . "\n\n";
    echo "XML Recebido (última):\n" . ($lastResp ?: '(sem)') . "\n";
}
