<?php
declare(strict_types=1);
mb_internal_encoding('UTF-8');
error_reporting(E_ALL);
ini_set('display_errors', '1');
date_default_timezone_set('America/Sao_Paulo');

/**
 * Consulta do Resultado do Lote GNRE (ResultadoLote v2 com pedido de consulta v1)
 * - Formulário (recibo, produção/homolog, salvar PDF)
 * - Execução e mensagem curta (status + linha digitável)
 * - Botão para baixar PDF (quando vier/salvar)
 * - XMLs em área colapsável
 * - Parser robusto: normalização do SOAP + REGEX à prova de namespace
 */

/* ===== CERTIFICADO ===== */
$pfxPath     = __DIR__ . '/../photopaper.pfx';
$pfxPassword = 'photo008A';

/* ===== Helpers ===== */
function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
function base_href(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $self   = $_SERVER['PHP_SELF'] ?? '';
    $dir    = rtrim(str_replace('\\', '/', dirname($self)), '/');
    return "{$scheme}://{$host}{$dir}";
}
/** Recorta do primeiro `<?xml` até o **último** `</...Envelope>` e remove BOM/lixos */
function normalizeSoap(string $raw): string {
    if ($raw === '') return $raw;
    $raw = preg_replace('/^\xEF\xBB\xBF/', '', $raw); // BOM
    $start = strpos($raw, '<?xml');
    if ($start === false) $start = 0;
    $slice = substr($raw, $start);
    if (preg_match_all('~</([A-Za-z0-9_:]*Envelope)>~', $slice, $m, PREG_OFFSET_CAPTURE)) {
        $last = end($m[0]);
        $endPos = $last[1] + strlen($last[0]);
        $slice = substr($slice, 0, $endPos);
    }
    return $slice;
}

/** Parser POR REGEX (independente de namespace) */
function parse_campos_basicos(string $xml): array {
    $out = [
        'statusCod'        => '',
        'statusMsg'        => '',
        'linhaDigitavel'   => '',
        'nossoNumero'      => '',
        'identificadorGuia'=> '',
    ];
    $src = normalizeSoap($xml);

    // codigo (situacaoProcess/codigo)
    if (preg_match('~<(?:\w+:)?codigo>\s*(\d+)\s*</(?:\w+:)?codigo>~i', $src, $m)) {
        $out['statusCod'] = $m[1];
    }
    // descricao
    if (preg_match('~<(?:\w+:)?descricao>\s*(.*?)\s*</(?:\w+:)?descricao>~is', $src, $m)) {
        $out['statusMsg'] = html_entity_decode(trim($m[1]), ENT_QUOTES | ENT_XML1, 'UTF-8');
    }
    // linhaDigitavel
    if (preg_match('~<(?:\w+:)?linhaDigitavel>\s*([0-9 ]+)\s*</(?:\w+:)?linhaDigitavel>~i', $src, $m)) {
        $out['linhaDigitavel'] = preg_replace('~\s+~', '', $m[1]); // sem espaços
    }
    // nossoNumero
    if (preg_match('~<(?:\w+:)?nossoNumero>\s*([0-9A-Za-z\-\.]+)\s*</(?:\w+:)?nossoNumero>~i', $src, $m)) {
        $out['nossoNumero'] = trim($m[1]);
    }
    // identificadorGuia
    if (preg_match('~<(?:\w+:)?identificadorGuia>\s*([0-9A-Za-z\-\.]+)\s*</(?:\w+:)?identificadorGuia>~i', $src, $m)) {
        $out['identificadorGuia'] = trim($m[1]);
    }
    return $out;
}

/** Extrai e salva TODOS PDFs presentes (pdfGuias|pdf) – regex, sem depender de SimpleXML */
function salvar_pdfs_regex(string $xml, string $logDir, string $recibo, ?string $prefer = null): array {
    $salvos = [];
    $src = normalizeSoap($xml);

    // Captura múltiplos nós pdfGuias/pdf (indep. de namespace). Permite quebras de linha.
    if (preg_match_all('~<(?:\w+:)?pdfGuias>\s*([A-Za-z0-9+/=\r\n]+)\s*</(?:\w+:)?pdfGuias>~i', $src, $m1)) {
        foreach ($m1[1] as $i => $b64) {
            $salvos[] = $b64;
        }
    }
    if (preg_match_all('~<(?:\w+:)?pdf>\s*([A-Za-z0-9+/=\r\n]+)\s*</(?:\w+:)?pdf>~i', $src, $m2)) {
        foreach ($m2[1] as $i => $b64) {
            $salvos[] = $b64;
        }
    }

    $files = [];
    $idx = 1;
    foreach ($salvos as $b64) {
        // Remove espaços/quebras apenas por segurança
        $b64 = trim($b64);
        // decode NÃO-STRICT para tolerar quebras/espacos
        $bin = base64_decode($b64, false);
        if ($bin === false || strlen($bin) < 100) continue;

        if ($prefer && $idx === 1) {
            $safe = preg_replace('~[^A-Za-z0-9_.-]+~', '_', $prefer);
            $fname = rtrim($logDir, '/') . '/' . $safe . '.pdf';
        } else {
            $fname = sprintf('%s/guia_%s_%02d.pdf', rtrim($logDir, '/'), $recibo, $idx);
        }
        if (@file_put_contents($fname, $bin) !== false) {
            $files[] = $fname;
            $idx++;
        }
    }
    return $files;
}

/* ===== Entrada (form) ===== */
$method    = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$recibo    = '';
$isProd    = true;   // Produção marcado por padrão
$savePdf   = true;   // Salvar PDF marcado por padrão

$statusCod = '';
$statusMsg = '';
$linhaDigitavel = '';
$pdfLinks  = [];
$xmlEnviado = '';
$xmlRecebido = '';
$guiaNossoNumero = '';
$guiaIdentificador = '';

$baseDir = dirname(__DIR__, 1);
$logDir  = $baseDir . '/logs';
if (!is_dir($logDir)) { @mkdir($logDir, 0775, true); }

if ($method === 'POST') {
    $recibo  = isset($_POST['recibo']) ? preg_replace('/\D+/', '', (string)$_POST['recibo']) : '';
    $isProd  = !empty($_POST['is_prod']);
    $savePdf = !empty($_POST['save_pdf']);

    if ($recibo === '') {
        $statusMsg = 'Informe o número do recibo.';
    } else {
        $ambiente  = $isProd ? 1 : 2;
        $serviceNS = 'http://www.gnre.pe.gov.br/webservice/GnreResultadoLote';
        $url       = $isProd
            ? 'https://www.gnre.pe.gov.br/gnreWS/services/GnreResultadoLote'
            : 'https://www.testegnre.pe.gov.br:444/gnreWS/services/GnreResultadoLote';

        // Pedido consulta (v1)
        $xml_consulta  = "<TConsLote_GNRE xmlns=\"http://www.gnre.pe.gov.br\">\n";
        $xml_consulta .= "    <ambiente>{$ambiente}</ambiente>\n";
        $xml_consulta .= "    <numeroRecibo>{$recibo}</numeroRecibo>\n";
        if ($savePdf) {
            $xml_consulta .= "    <incluirPDFGuias>S</incluirPDFGuias>\n";
            $xml_consulta .= "    <incluirArquivoPagamento>S</incluirArquivoPagamento>\n";
        } else {
            $xml_consulta .= "    <incluirPDFGuias>N</incluirPDFGuias>\n";
            $xml_consulta .= "    <incluirArquivoPagamento>N</incluirArquivoPagamento>\n";
        }
        $xml_consulta .= "</TConsLote_GNRE>";

        // Envelope SOAP 1.2 (bare)
        $soapEnvelope = <<<SOAP
<?xml version="1.0" encoding="UTF-8"?>
<soap12:Envelope xmlns:soap12="http://www.w3.org/2003/05/soap-envelope" xmlns:ws="{$serviceNS}">
  <soap12:Header>
    <ws:gnreCabecMsg><ws:versaoDados>2.00</ws:versaoDados></ws:gnreCabecMsg>
  </soap12:Header>
  <soap12:Body><ws:gnreDadosMsg>
{$xml_consulta}
  </ws:gnreDadosMsg></soap12:Body>
</soap12:Envelope>
SOAP;

        $ts        = date('Ymd_His');
        $prefixLog = $logDir . "/consultaLote_{$recibo}_{$ts}";
        @file_put_contents("{$prefixLog}_enviado.xml", $soapEnvelope);
        $xmlEnviado = $soapEnvelope;

        $headers = [
          // Em SOAP 1.2 o action pode ir no Content-Type; manter os dois não atrapalha:
          'Content-Type: application/soap+xml; charset=utf-8; action="http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar"',
          'SOAPAction: "http://www.gnre.pe.gov.br/webservice/GnreResultadoLote/consultar"',
        ];

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST            => true,
            CURLOPT_POSTFIELDS      => $soapEnvelope,
            CURLOPT_HTTPHEADER      => $headers,
            CURLOPT_RETURNTRANSFER  => true,
            CURLOPT_HEADER          => false,
            CURLOPT_TIMEOUT         => 60,
            CURLOPT_SSL_VERIFYPEER  => true,
            CURLOPT_SSL_VERIFYHOST  => 2,
            CURLOPT_SSLCERT         => $pfxPath,
            CURLOPT_SSLCERTTYPE     => 'P12',
            CURLOPT_SSLCERTPASSWD   => $pfxPassword,
        ]);
        $response = curl_exec($ch);
        $error    = curl_error($ch);
        $httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $xmlRecebido = $response ?: $error;
        @file_put_contents("{$prefixLog}_recebido.xml", $xmlRecebido);

        if ($response) {
            // ===== Parser POR REGEX (blindado pra namespace) =====
            $parsed = parse_campos_basicos($response);
            $statusCod         = $parsed['statusCod'];
            $statusMsg         = $parsed['statusMsg'];
            $linhaDigitavel    = $parsed['linhaDigitavel'];
            $guiaNossoNumero   = $parsed['nossoNumero'];
            $guiaIdentificador = $parsed['identificadorGuia'];

            // ===== Salvar PDFs (se marcado) =====
            if ($savePdf) {
                $prefer = '';
                if ($guiaNossoNumero)       $prefer = 'GNRE_' . preg_replace('~\D+~', '', $guiaNossoNumero);
                elseif ($guiaIdentificador)  $prefer = 'GNRE_' . preg_replace('~\D+~', '', $guiaIdentificador);

                $savedFiles = salvar_pdfs_regex($response, $logDir, $recibo, $prefer ?: null);
                if ($savedFiles) {
                    $base = base_href();
                    foreach ($savedFiles as $abs) {
                        $fileName = basename($abs);
                        // OBS: /logs está UM nível acima do script; ajuste o path público conforme seu deploy.
                        $pdfLinks[] = [
                            'name' => $fileName,
                            'url'  => $base . '/../logs/' . rawurlencode($fileName),
                        ];
                    }
                }
            }

        } else {
            $statusMsg = $error ?: 'Falha ao conectar.';
        }
    }
}

/* ===== UI ===== */
?>
<!doctype html>
<html lang="pt-br">
<head>
<meta charset="utf-8">
<title>Consulta GNRE - Resultado do Lote</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
  :root{color-scheme:dark light}
  body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu,"Helvetica Neue",Arial,sans-serif;background:#0b0b0c;color:#e6e6e6;margin:0;padding:24px}
  .wrap{max-width:960px;margin:0 auto}
  .card{background:#151517;border:1px solid #2a2a2f;border-radius:14px;padding:20px;margin:0 0 16px}
  .row{display:flex;flex-wrap:wrap;gap:12px;align-items:center}
  label{display:block;font-size:14px;color:#cbd5e0;margin-bottom:6px}
  input[type="text"]{width:100%;padding:10px;border-radius:10px;border:1px solid #3a3a40;background:#0f0f11;color:#fff}
  .check{display:flex;align-items:center;gap:8px}
  .btn{display:inline-block;padding:10px 14px;border-radius:10px;border:1px solid #3a3a40;text-decoration:none;color:#fff;background:#2b6cb0;cursor:pointer}
  .btn:hover{background:#2c5282}
  .btn:disabled{opacity:.5;cursor:not-allowed}
  .warn{color:#fbd38d}
  .mono{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,"Courier New",monospace}
  details summary{cursor:pointer;color:#a0aec0}
  textarea{width:100%;height:220px;border-radius:10px;border:1px solid #3a3a40;background:#0f0f11;color:#fff;padding:10px;resize:vertical}
</style>
</head>
<body>
<div class="wrap">

  <div class="card">
    <h2 style="margin:0 0 12px">Consulta do Resultado do Lote – GNRE</h2>
    <form method="post" class="row" action="">
      <div style="flex:1 1 260px;min-width:260px">
        <label for="recibo">Nº do recibo</label>
        <input type="text" id="recibo" name="recibo" required value="<?=h($recibo)?>" placeholder="Ex.: 25000225744095" class="mono">
      </div>
      <div style="flex:1 1 220px;min-width:220px">
        <label>&nbsp;</label>
        <div class="check">
          <input type="checkbox" id="is_prod" name="is_prod" <?= $isProd?'checked':''?>>
          <label for="is_prod" style="margin:0">Produção (desmarque para Homolog)</label>
        </div>
        <div class="check" style="margin-top:8px">
          <input type="checkbox" id="save_pdf" name="save_pdf" <?= $savePdf?'checked':''?>>
          <label for="save_pdf" style="margin:0">Salvar PDF (quando disponível)</label>
        </div>
      </div>
      <div style="flex:0 0 auto;align-self:flex-end">
        <button class="btn" type="submit">Consultar</button>
      </div>
    </form>
  </div>

  <?php if ($method === 'POST'): ?>
  <div class="card">
    <h3 style="margin-top:0">Resultado</h3>
    <?php if ($statusCod !== ''): ?>
      <p>Status do lote: <span class="mono"><?=h($statusCod)?></span> — <span><?=h($statusMsg)?></span></p>
    <?php else: ?>
      <p class="warn">Sem código de status no retorno.</p>
    <?php endif; ?>

    <?php if ($linhaDigitavel): ?>
      <p>🎯 Linha digitável para pagamento:</p>
      <p class="mono" style="font-size:18px"><?=h($linhaDigitavel)?></p>
    <?php else: ?>
      <p class="warn">Linha digitável não veio no retorno.</p>
    <?php endif; ?>

    <div style="margin-top:12px" class="row">
      <?php if (!empty($pdfLinks)): ?>
        <?php foreach ($pdfLinks as $d): ?>
          <a class="btn" target="_blank" href="<?=h($d['url'])?>" download="<?=h($d['name'])?>">Visualizar/baixar PDF</a>
        <?php endforeach; ?>
      <?php else: ?>
        <button class="btn" type="button" disabled>PDF indisponível</button>
      <?php endif; ?>
    </div>
  </div>

  <div class="card">
    <details>
      <summary>XMLs (mostrar/ocultar)</summary>
      <div style="margin-top:12px">
        <label>XML Enviado</label>
        <textarea readonly class="mono"><?=h($xmlEnviado)?></textarea>
      </div>
      <div style="margin-top:12px">
        <label>XML Recebido</label>
        <textarea readonly class="mono"><?=h(normalizeSoap($xmlRecebido))?></textarea>
      </div>
    </details>
  </div>
  <?php endif; ?>

</div>
</body>
</html>
